package nl.ivojonker.icn.configuration.services;

import java.io.InputStream;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import nl.ivojonker.icn.configuration.Configuration;
import nl.ivojonker.icn.configuration.ConfigurationJSON;

import com.ibm.ecm.configuration.Config;
import com.ibm.ecm.extension.PluginService;
import com.ibm.ecm.extension.PluginServiceCallbacks;
import com.ibm.json.java.JSONArray;
import com.ibm.json.java.JSONObject;

/**
 * The ConfigurationService handles the retrieval of configuration and status, and serves as a proxy in creating the required configuration table.
 * 
 * @author nl.ivojonker
 */
public class ConfigurationService  extends PluginService {
	
	private static Logger logger = Logger.getLogger(ConfigurationService.class.getCanonicalName());

	/**
	 * {@inheritDoc}
	 */
	public String getId() {
		return "ConfigurationService";
	}

	/**
	 * not used / nothing overridden.
	 */
	public String getOverriddenService() {
		return null;
	}

	/**
	 * Executes a specific instruction(provided as parameter). Some actions require administrator privileges.
	 * instructions:
	 * config -> returns configuration; primarily used for frontend dojo/CentralConfiguration
	 * status -> returns status; only used on the configuration pane
	 * createtable/droptable; for creating / dropping the table in the configured datasource. only used on the configuration pane
	 * persist -> for persisting configuration.
	 */
	public void execute(PluginServiceCallbacks callbacks,
			HttpServletRequest request, HttpServletResponse response)
			throws Exception {
		String instruction = request.getParameter("instruction");
		
		if ("config".equals(instruction)){
			response.getWriter().append(ConfigurationJSON.getConfigAsJSONArray(true).toString());
		}else if (isAdminUser(request)){
			if ("status".equals(instruction) || instruction == null){
				response.getWriter().append(getStatus().toString());
			}else if ("createtable".equals(instruction)){
				JSONObject _response = new JSONObject();
				try{
					Configuration.getInstanceSupressExceptions(false).createTable();
					_response.put("response", "Table succesfully created.");
					Configuration.getInstance(true);
				}catch(Exception any){
					_response = new JSONObject();
					_response.put("response","error:" + any.getMessage());
				}finally{
					response.getWriter().append(_response.toString());
				}
			}else if ("droptable".equals(instruction)){
				JSONObject _response = new JSONObject();
				try{
					Configuration.getInstanceSupressExceptions(false).dropTable();
					_response.put("response", "Table sucesfully dropped.");
					Configuration.getInstance(true);
				}catch(Exception any){
					_response = new JSONObject();
					_response.put("response","error:" + any.getMessage());
				}finally{
					response.getWriter().append(_response.toString());
				}
			}else if ("persist".equals(instruction)){
				try{
					JSONArray configToPersist = JSONArray.parse(request.getParameter("persist"));
					ConfigurationJSON.persistConfigFromJSONArray(configToPersist);
				}catch(Exception ignore){
					;
				}finally{
					response.getWriter().append(getStatus().toString());
				}
			}else{
				response.sendError(HttpServletResponse.SC_BAD_REQUEST);
			}
		}else{
			response.sendError(HttpServletResponse.SC_FORBIDDEN);
		}

	}
	
	/**
	 * Used for the plugin/configuration screen to show status and config.
	 * 
	 * @return json containing confoigured tablename, JNDI datasource, configuration and status.
	 */
	private JSONObject getStatus(){
		Properties properties = new Properties();
		try{
			InputStream propertyFile = this.getClass().getClassLoader().getResourceAsStream("config.properties");
			properties.load(propertyFile);
		}catch(Exception any){
			logger.log(Level.SEVERE,"Failure while trying to read plugin.jar!config.properties.",any);
		}
		
		JSONObject result = new JSONObject();

		result.put("tablename",properties.getProperty("CONFIGTABLENAME"));
		result.put("jndi",properties.getProperty("DATASOURCEJNDI"));
		try{
			result.put("config", ConfigurationJSON.getConfigAsJSONArray(false));
			result.put("status", "ready");
		}catch(Exception any){
			result.put("status", "failure:"+any.getMessage());
			result.put("config", new JSONArray());
			logger.log(Level.SEVERE,"Failure while trying to read config.",any);
		}
		return result;
	}
	
	
	/**
	 * Checks if the current user is admin user.
	 * TODO: Function possibly not future proof in case group-support is added as adminUsers setting.
	 * @param request containing a valid session
	 * @return 
	 */
	private boolean isAdminUser(HttpServletRequest request){
		try{
			String currentUser=Config.getUserConfig(request).getObjectId();

			if (currentUser==null || currentUser.trim().length()==0)
				return false;

			for (String configuredAdmin : Config.getSettingsConfig("navigator").getAdminUsers()){
				if (currentUser.equalsIgnoreCase(configuredAdmin))
					return true;
			}
		}catch(Exception any){
			logger.log(Level.WARNING,"Failed in determining if logged-in-user is admin-user when accesing the central configuration plugin.",any);
		}
		return false;
	}
	
	
	
}
